<?php
/**
 * WarForms Reboot - Site Health integration (Step 18)
 *
 * Adds a Site Health "Info" section and a few lightweight tests to help diagnose
 * DB/schema issues without exposing sensitive data.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Internal: check whether the entries table exists.
 *
 * @return bool
 */
function warforms_reboot_entries_table_exists(): bool {
	if ( ! function_exists( 'warforms_reboot_entries_table' ) ) {
		return false;
	}

	global $wpdb;
	$table = warforms_reboot_entries_table();
	if ( ! is_string( $table ) || $table === '' ) {
		return false;
	}

	$found = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) );
	return (string) $found === (string) $table;
}

/**
 * Site Health -> Info panel details.
 */
add_filter( 'debug_information', function( array $info ): array {
	$stored_schema = (int) get_option( 'warforms_reboot_schema_version', 0 );
	$expected_schema = defined( 'WARFORMS_REBOOT_DB_VERSION' ) ? (int) WARFORMS_REBOOT_DB_VERSION : 0;

	$table_name = function_exists( 'warforms_reboot_entries_table' ) ? warforms_reboot_entries_table() : '(unknown)';
	$table_exists = warforms_reboot_entries_table_exists() ? 'Yes' : 'No';

	$last = function_exists( 'warforms_reboot_get_last_error' ) ? warforms_reboot_get_last_error() : [];
	$last_code = isset( $last['code'] ) ? (string) $last['code'] : '';
	$last_detail = isset( $last['detail'] ) ? (string) $last['detail'] : '';
	$last_time = isset( $last['time'] ) ? (int) $last['time'] : 0;

	$last_time_human = $last_time ? date_i18n( 'Y-m-d H:i:s', $last_time ) : '—';
	$last_code_human = $last_code !== '' ? $last_code : '—';
	$last_detail_human = $last_detail !== '' ? $last_detail : '—';

	$info['warforms_reboot'] = [
		'label'       => 'WarForms Reboot',
		'description' => 'Diagnostics for the WarForms Reboot plugin.',
		'fields'      => [
			'plugin_version' => [
				'label' => 'Plugin version',
				'value' => defined( 'WARFORMS_REBOOT_VERSION' ) ? WARFORMS_REBOOT_VERSION : '—',
			],
			'wp_version' => [
				'label' => 'WordPress version',
				'value' => get_bloginfo( 'version' ),
			],
			'php_version' => [
				'label' => 'PHP version',
				'value' => PHP_VERSION,
			],
			'entries_table' => [
				'label' => 'Entries table',
				'value' => $table_name,
			],
			'entries_table_exists' => [
				'label' => 'Entries table exists',
				'value' => $table_exists,
			],
			'schema_version_expected' => [
				'label' => 'Expected schema version',
				'value' => $expected_schema ? (string) $expected_schema : '—',
			],
			'schema_version_stored' => [
				'label' => 'Stored schema version',
				'value' => (string) $stored_schema,
			],
			'last_error_code' => [
				'label' => 'Last error code',
				'value' => $last_code_human,
			],
			'last_error_time' => [
				'label' => 'Last error time',
				'value' => $last_time_human,
			],
			'last_error_detail' => [
				'label'   => 'Last error detail',
				'value'   => $last_detail_human,
				'private' => true,
			],
		],
	];

	return $info;
} );

/**
 * Site Health -> Status tests.
 */
add_filter( 'site_status_tests', function( array $tests ): array {
	$tests['direct']['warforms_reboot_entries_table'] = [
		'label' => 'WarForms entries table',
		'test'  => 'warforms_reboot_site_health_test_entries_table',
	];

	$tests['direct']['warforms_reboot_schema_version'] = [
		'label' => 'WarForms schema version',
		'test'  => 'warforms_reboot_site_health_test_schema_version',
	];

	$tests['direct']['warforms_reboot_last_error'] = [
		'label' => 'WarForms last error',
		'test'  => 'warforms_reboot_site_health_test_last_error',
	];

	return $tests;
} );

function warforms_reboot_site_health_badge(): array {
	return [
		'label' => 'WarForms',
		'color' => 'blue',
	];
}

function warforms_reboot_site_health_diagnostics_link(): string {
	if ( ! is_admin() ) {
		return '';
	}
	$url = admin_url( 'edit.php?post_type=warform&page=warforms-reboot-diagnostics' );
	return '<p><a href="' . esc_url( $url ) . '">Open WarForms Diagnostics</a></p>';
}

function warforms_reboot_site_health_test_entries_table(): array {
	$exists = warforms_reboot_entries_table_exists();

	return [
		'label'       => $exists ? 'Entries table exists.' : 'Entries table is missing.',
		'status'      => $exists ? 'good' : 'critical',
		'badge'       => warforms_reboot_site_health_badge(),
		'description' => $exists
			? '<p>WarForms entries table was found.</p>'
			: '<p>WarForms entries table is missing. Reactivate the plugin or check database permissions.</p>',
		'actions'     => $exists ? '' : warforms_reboot_site_health_diagnostics_link(),
		'test'        => 'warforms_reboot_entries_table',
	];
}

function warforms_reboot_site_health_test_schema_version(): array {
	$stored = (int) get_option( 'warforms_reboot_schema_version', 0 );
	$expected = defined( 'WARFORMS_REBOOT_DB_VERSION' ) ? (int) WARFORMS_REBOOT_DB_VERSION : 0;

	$ok = ( $expected > 0 && $stored === $expected );

	$desc = '<p>Expected schema version: <code>' . esc_html( (string) $expected ) . '</code>. Stored: <code>' . esc_html( (string) $stored ) . '</code>.</p>';
	if ( ! $ok ) {
		$desc .= '<p>If these do not match, WarForms will attempt to run its installer on load. If it persists, check DB permissions.</p>';
	}

	return [
		'label'       => $ok ? 'Schema version is up to date.' : 'Schema version is out of date.',
		'status'      => $ok ? 'good' : 'recommended',
		'badge'       => warforms_reboot_site_health_badge(),
		'description' => $desc,
		'actions'     => $ok ? '' : warforms_reboot_site_health_diagnostics_link(),
		'test'        => 'warforms_reboot_schema_version',
	];
}

function warforms_reboot_site_health_test_last_error(): array {
	$last = function_exists( 'warforms_reboot_get_last_error' ) ? warforms_reboot_get_last_error() : [];
	$code = isset( $last['code'] ) ? (string) $last['code'] : '';
	$time = isset( $last['time'] ) ? (int) $last['time'] : 0;

	if ( $code === '' ) {
		return [
			'label'       => 'No recent WarForms errors recorded.',
			'status'      => 'good',
			'badge'       => warforms_reboot_site_health_badge(),
			'description' => '<p>WarForms has not recorded a last_error.</p>',
			'actions'     => '',
			'test'        => 'warforms_reboot_last_error',
		];
	}

	$age = $time ? ( time() - $time ) : 0;
	$recent = $age > 0 && $age < DAY_IN_SECONDS;

	$desc = '<p>Last error code: <code>' . esc_html( $code ) . '</code>.</p>';
	if ( $time ) {
		$desc .= '<p>Time: <code>' . esc_html( date_i18n( 'Y-m-d H:i:s', $time ) ) . '</code>.</p>';
	}
	$desc .= '<p>See WarForms Diagnostics for details.</p>';

	return [
		'label'       => $recent ? 'Recent WarForms error recorded.' : 'WarForms error recorded (older).',
		'status'      => $recent ? 'recommended' : 'good',
		'badge'       => warforms_reboot_site_health_badge(),
		'description' => $desc,
		'actions'     => warforms_reboot_site_health_diagnostics_link(),
		'test'        => 'warforms_reboot_last_error',
	];
}
